using System;
using System.Collections;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using System.Text.RegularExpressions;
using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.GUI.controls
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/13/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Class which allows for the entry of a unit id and product code via
	///a scan or a manual entry.  The product type, unit status, donation type
	///restricted for and blood label will all be populated and read only to reflect the corresponding
	///information for the given unit id and product code.
	///</summary>
	#endregion

	public class BloodUnitSelector : GUI.controls.BaseControl
	{
		#region Form Variables

		private Common.AdditionalBloodUnitSearchCriteria _searchCriteria = Common.AdditionalBloodUnitSearchCriteria.UnitsInFinalStatus;

		//private BOL.BloodUnit _bloodUnit;
		private BOL.BloodUnit _currentBloodUnit;
		private System.Data.DataTable _unitTable;
	
		private Regex _isbtUnitID;
		private Regex _isbtProductCode;
		private Regex _codabarUnitID;
		private Regex _codabarProductCode;
		
		/// <summary>
		/// Variable to keep recursive focus from occurring when changing focus in a leave event; A value of true indicates
		/// that the form has already started processing the leave event and it shouldn't try to do it again
		/// </summary>
		private bool _recursionGuard = false;

		#endregion

		#region Delegates
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// An event that clients can use to be notified whenever a unit is selected
		/// </summary>
		public delegate void SelectBtnClickedEventHandler(object sender, EventArgs e);
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// Select button click event
		/// </summary>
		public event SelectBtnClickedEventHandler SelectBtnClick;
		//
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// An event that clients can use to be notified whenever a unit is cleared
		/// </summary>
		public delegate void ClearBtnClickedEventHandler(object sender, EventArgs e);
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// Clear button click event
		/// </summary>
		public event ClearBtnClickedEventHandler ClearBtnClick;
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// An event that clients can use to be notified whenever a unit is updated
		/// </summary>
		public delegate void UnitUpdatedEventHandler(object sender, EventArgs e);
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///
		/// <summary>
		/// Unit updated event
		/// </summary>
		public event UnitUpdatedEventHandler UnitUpdate;
		//
		private System.EventHandler onSearchClick;
		#endregion

		#region Form Controls

		private System.Windows.Forms.GroupBox grpSearch;
		private System.Windows.Forms.Button btnSelectUnit;
		private System.Windows.Forms.PictureBox pictureBox2;
		private System.Windows.Forms.PictureBox pictureBox4;
		private System.Windows.Forms.Label lblUnitID;
		private System.Windows.Forms.Label lblProduct;
		private System.Windows.Forms.GroupBox grpImage;
		private System.Windows.Forms.PictureBox pictureBox1;
		private System.Windows.Forms.Button btnSelect;
		private System.Windows.Forms.Button btnClear;
		/// <summary>
		/// Product Code text box
		/// </summary>
		public gov.va.med.vbecs.GUI.controls.ProductCodeTextbox txtProductID;
		/// <summary>
		/// Unit ID text box
		/// </summary>
		public gov.va.med.vbecs.GUI.controls.UnitIDTextbox txtUnitID;
		private gov.va.med.vbecs.GUI.controls.VbecsErrorProvider errorProvider1; // **** 508 Compliance ***
		private gov.va.med.vbecs.GUI.controls.BloodUnitLabel bloodUnitLabel1;
		private gov.va.med.vbecs.GUI.controls.BloodUnitDetailsTab bloodUnitDetailsTab1;
		private System.ComponentModel.IContainer components;
		#endregion

		#region Constructors/Destructors
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		///<summary>
		///BloodUnitSelector class constructor
		///Sets the blood unit associated with the control
		///</summary>	
		public BloodUnitSelector()
		{

			InitializeComponent();
	
//bb		this._currentBloodUnit = new BloodUnit(); 
			//
			this._codabarProductCode = Common.RegularExpressions.CodabarProductCode();
			this._codabarUnitID = Common.RegularExpressions.PooledCodabarEyeReadableUnitId();
			this._isbtProductCode = Common.RegularExpressions.ISBT128ProductCode();
			this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();

		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		/// <summary>
		/// BloodUnitSelector class constructor
		/// Sets the blood unit associated with the control, adds the the control to the container
		/// </summary>
		/// <param name="container">Container</param>
		public BloodUnitSelector(System.ComponentModel.IContainer container)
		{
			container.Add(this);
			InitializeComponent();
			try
			{
//bb				this._currentBloodUnit = new BloodUnit(); 
				//
				this._codabarProductCode = Common.RegularExpressions.CodabarProductCode();
				this._codabarUnitID = Common.RegularExpressions.PooledCodabarEyeReadableUnitId();
				this._isbtProductCode = Common.RegularExpressions.ISBT128ProductCode();
				this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();
			}
			catch(Exception exp)
			{
				MessageBox.Show(exp.Message);
			}

		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}
		#endregion

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(BloodUnitSelector));
			this.grpSearch = new System.Windows.Forms.GroupBox();
			this.txtProductID = new gov.va.med.vbecs.GUI.controls.ProductCodeTextbox(this.components);
			this.txtUnitID = new gov.va.med.vbecs.GUI.controls.UnitIDTextbox(this.components);
			this.btnSelectUnit = new System.Windows.Forms.Button();
			this.pictureBox2 = new System.Windows.Forms.PictureBox();
			this.pictureBox4 = new System.Windows.Forms.PictureBox();
			this.lblUnitID = new System.Windows.Forms.Label();
			this.lblProduct = new System.Windows.Forms.Label();
			this.grpImage = new System.Windows.Forms.GroupBox();
			this.pictureBox1 = new System.Windows.Forms.PictureBox();
			this.btnSelect = new System.Windows.Forms.Button();
			this.btnClear = new System.Windows.Forms.Button();
			this.errorProvider1 = new gov.va.med.vbecs.GUI.controls.VbecsErrorProvider();
			this.bloodUnitLabel1 = new gov.va.med.vbecs.GUI.controls.BloodUnitLabel(this.components);
			this.bloodUnitDetailsTab1 = new gov.va.med.vbecs.GUI.controls.BloodUnitDetailsTab();
			this.grpSearch.SuspendLayout();
			this.grpImage.SuspendLayout();
			this.SuspendLayout();
			// 
			// grpSearch
			// 
			this.grpSearch.AccessibleDescription = "Unit Search";
			this.grpSearch.AccessibleName = "Unit Search";
			this.grpSearch.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpSearch.Controls.Add(this.txtProductID);
			this.grpSearch.Controls.Add(this.txtUnitID);
			this.grpSearch.Controls.Add(this.btnSelectUnit);
			this.grpSearch.Controls.Add(this.pictureBox2);
			this.grpSearch.Controls.Add(this.pictureBox4);
			this.grpSearch.Controls.Add(this.lblUnitID);
			this.grpSearch.Controls.Add(this.lblProduct);
			this.grpSearch.Location = new System.Drawing.Point(44, 0);
			this.grpSearch.Name = "grpSearch";
			this.grpSearch.Size = new System.Drawing.Size(240, 64);
			this.grpSearch.TabIndex = 0;
			this.grpSearch.TabStop = false;
			this.grpSearch.Text = "Unit Search";
			// 
			// txtProductID
			// 
			this.txtProductID.AccessibleDescription = "Product";
			this.txtProductID.AccessibleName = "Product";
			this.txtProductID.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtProductID.Barcode = false;
			this.txtProductID.BarcodeType = gov.va.med.vbecs.Common.BarcodeType.None;
			this.txtProductID.Location = new System.Drawing.Point(88, 40);
			this.txtProductID.MaxLength = 10;
			this.txtProductID.Name = "txtProductID";
			this.txtProductID.RecursionGuard = false;
			this.txtProductID.Size = new System.Drawing.Size(124, 20);
			this.txtProductID.TabIndex = 2;
			this.txtProductID.Text = "";
			this.txtProductID.KeyDown += new System.Windows.Forms.KeyEventHandler(this.SearchKeyDown);
			this.txtProductID.Leave += new System.EventHandler(this.txtProductID_Leave);
			// 
			// txtUnitID
			// 
			this.txtUnitID.AccessibleDescription = "Unit I D";
			this.txtUnitID.AccessibleName = "Unit I D";
			this.txtUnitID.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtUnitID.Barcode = false;
			this.txtUnitID.BarcodeType = gov.va.med.vbecs.Common.BarcodeType.None;
			this.txtUnitID.Location = new System.Drawing.Point(88, 16);
			this.txtUnitID.MaxLength = 15;
			this.txtUnitID.Name = "txtUnitID";
			this.txtUnitID.Size = new System.Drawing.Size(124, 20);
			this.txtUnitID.TabIndex = 1;
			this.txtUnitID.Text = "";
			this.txtUnitID.KeyDown += new System.Windows.Forms.KeyEventHandler(this.SearchKeyDown);
			this.txtUnitID.Leave += new System.EventHandler(this.txtUnitID_Leave);
			// 
			// btnSelectUnit
			// 
			this.btnSelectUnit.AccessibleDescription = "find";
			this.btnSelectUnit.AccessibleName = "find";
			this.btnSelectUnit.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.btnSelectUnit.Font = new System.Drawing.Font("Arial", 5.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnSelectUnit.Location = new System.Drawing.Point(212, 16);
			this.btnSelectUnit.Name = "btnSelectUnit";
			this.btnSelectUnit.Size = new System.Drawing.Size(24, 20);
			this.btnSelectUnit.TabIndex = 3;
			this.btnSelectUnit.Text = "find";
			this.btnSelectUnit.Click += new System.EventHandler(this.btnSelectUnit_Click);
			// 
			// pictureBox2
			// 
			this.pictureBox2.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox2.Image")));
			this.pictureBox2.Location = new System.Drawing.Point(8, 16);
			this.pictureBox2.Name = "pictureBox2";
			this.pictureBox2.Size = new System.Drawing.Size(20, 20);
			this.pictureBox2.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.pictureBox2.TabIndex = 18;
			this.pictureBox2.TabStop = false;
			// 
			// pictureBox4
			// 
			this.pictureBox4.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox4.Image")));
			this.pictureBox4.Location = new System.Drawing.Point(8, 40);
			this.pictureBox4.Name = "pictureBox4";
			this.pictureBox4.Size = new System.Drawing.Size(20, 20);
			this.pictureBox4.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.pictureBox4.TabIndex = 16;
			this.pictureBox4.TabStop = false;
			// 
			// lblUnitID
			// 
			this.lblUnitID.Location = new System.Drawing.Point(28, 16);
			this.lblUnitID.Name = "lblUnitID";
			this.lblUnitID.Size = new System.Drawing.Size(56, 20);
			this.lblUnitID.TabIndex = 0;
			this.lblUnitID.Text = "Unit ID*";
			this.lblUnitID.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// lblProduct
			// 
			this.lblProduct.Location = new System.Drawing.Point(28, 40);
			this.lblProduct.Name = "lblProduct";
			this.lblProduct.Size = new System.Drawing.Size(56, 20);
			this.lblProduct.TabIndex = 2;
			this.lblProduct.Text = "Product*";
			this.lblProduct.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// grpImage
			// 
			this.grpImage.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left)));
			this.grpImage.Controls.Add(this.pictureBox1);
			this.grpImage.Location = new System.Drawing.Point(0, 0);
			this.grpImage.Name = "grpImage";
			this.grpImage.Size = new System.Drawing.Size(40, 208);
			this.grpImage.TabIndex = 5;
			this.grpImage.TabStop = false;
			// 
			// pictureBox1
			// 
			this.pictureBox1.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox1.Image")));
			this.pictureBox1.Location = new System.Drawing.Point(4, 12);
			this.pictureBox1.Name = "pictureBox1";
			this.pictureBox1.Size = new System.Drawing.Size(32, 32);
			this.pictureBox1.SizeMode = System.Windows.Forms.PictureBoxSizeMode.StretchImage;
			this.pictureBox1.TabIndex = 0;
			this.pictureBox1.TabStop = false;
			// 
			// btnSelect
			// 
			this.btnSelect.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnSelect.Enabled = false;
			this.btnSelect.Location = new System.Drawing.Point(126, 210);
			this.btnSelect.Name = "btnSelect";
			this.btnSelect.TabIndex = 5;
			this.btnSelect.Text = "&Select";
			this.btnSelect.Click += new System.EventHandler(this.btnSelect_Click);
			// 
			// btnClear
			// 
			this.btnClear.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnClear.Location = new System.Drawing.Point(206, 210);
			this.btnClear.Name = "btnClear";
			this.btnClear.TabIndex = 6;
			this.btnClear.Text = "C&lear";
			this.btnClear.Click += new System.EventHandler(this.btnClear_Click);
			// 
			// errorProvider1
			// 
			this.errorProvider1.ContainerControl = this;
			this.errorProvider1.DataMember = "";
			this.errorProvider1.Icon = ((System.Drawing.Icon)(resources.GetObject("errorProvider1.Icon")));
			// 
			// bloodUnitLabel1
			// 
			this.bloodUnitLabel1.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.bloodUnitLabel1.BackColor = System.Drawing.Color.WhiteSmoke;
			this.bloodUnitLabel1.BloodUnit = null;
			this.bloodUnitLabel1.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.bloodUnitLabel1.Location = new System.Drawing.Point(286, 4);
			this.bloodUnitLabel1.Name = "bloodUnitLabel1";
			this.bloodUnitLabel1.Size = new System.Drawing.Size(257, 204);
			this.bloodUnitLabel1.TabIndex = 2;
			this.bloodUnitLabel1.TabStop = false;
			// 
			// bloodUnitDetailsTab1
			// 
			this.bloodUnitDetailsTab1.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.bloodUnitDetailsTab1.BloodUnit = null;
			this.bloodUnitDetailsTab1.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.bloodUnitDetailsTab1.Location = new System.Drawing.Point(44, 64);
			this.bloodUnitDetailsTab1.Name = "bloodUnitDetailsTab1";
			this.bloodUnitDetailsTab1.Size = new System.Drawing.Size(240, 144);
			this.bloodUnitDetailsTab1.TabIndex = 4;
			// 
			// BloodUnitSelector
			// 
			this.Controls.Add(this.bloodUnitLabel1);
			this.Controls.Add(this.bloodUnitDetailsTab1);
			this.Controls.Add(this.btnClear);
			this.Controls.Add(this.btnSelect);
			this.Controls.Add(this.grpSearch);
			this.Controls.Add(this.grpImage);
			this.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.Name = "BloodUnitSelector";
			this.Size = new System.Drawing.Size(544, 236);
			this.Load += new System.EventHandler(this.BloodUnitSelector_Load);
			this.grpSearch.ResumeLayout(false);
			this.grpImage.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Properties

		/// <summary>
		/// Get/Set SelectButtonEnabled
		/// </summary>
		public bool SelectButtonEnabled
		{
			get
			{
				// **** 508 Compliance ***
				return this.btnSelect.Enabled;
			}
			set
			{
				this.btnSelect.Enabled = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		/// <summary> 
		/// Gets or sets docking style property. Editable at design time
		/// </summary>
		[
		Category("Layout"),
		Description("Docking style"),
		Bindable(false),
		Browsable(true),
		]
		public override DockStyle Dock 
		{
			get 
			{ 
				return base.Dock;
			}
			set 
			{ 
				base.Dock = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		/// <summary> 
		/// Gets or sets anchor style property. Editable at design time
		/// </summary>
		[
		Category("Layout"),
		Description("Anchor style"),
		Bindable(false),
		Browsable(true),
		]
		public override AnchorStyles Anchor
		{
			get 
			{ 
				return base.Anchor;
			}
			set 
			{ 
				base.Anchor = value;
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		/// <summary>
		/// Used to return units that meet additional
		/// criteria.  For example, also bring 
		/// back units in final status.  
		/// </summary>
		public Common.AdditionalBloodUnitSearchCriteria AdditionalSearchCriteria
		{
			get
			{
				return(this._searchCriteria);
			}
			set
			{
				this._searchCriteria = value;
			}
		}

		/// <summary>
		/// Get/Set BloodUnit
		/// </summary>
		[Browsable(false)]
		public BOL.BloodUnit BloodUnit
		{
			get 
			{
				return this._currentBloodUnit;	// ._bloodUnit; 
			}
			set 
			{
				if (this._currentBloodUnit != null)
				{
					this.UpdateGUIForBloodUnit();
				}
				else
				{
					this.Clear(false);
				}
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		/// <summary>
		/// Gets or sets table of selected units
		/// </summary>
		[Browsable(false)]
		public System.Data.DataTable UnitTable
		{
			get 
			{ 
				return _unitTable; 
			}
			set
			{
				_unitTable = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		/// <summary>
		/// Returns the eye readable unit id entered in the textbox field
		/// </summary>
		public string EnteredEyeReadableUnitId
		{
			get
			{
				return(this.txtUnitID.Text.Trim());
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		/// <summary>
		/// Returns the unit product code enetered in the textbox field - used to filter the UC5 list
		/// </summary>
		public string EnteredUnitProductCode
		{
			get
			{
				return(this.txtProductID.Text.Trim());
			}
		}

		#endregion

		#region Events

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		///
		/// <summary>
		/// The event that gets raise when the user clicks search
		/// </summary>
		public event EventHandler SearchClick
		{
			add 
			{ 
				onSearchClick += value; 
			}
			remove 
			{ 
				onSearchClick -= value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Invoke the select button click event
		/// </summary>
		/// <param name="e">Event args</param>
		protected virtual void OnSelectBtnClick(EventArgs e) 
		{
			if (SelectBtnClick != null) 
			{
				SelectBtnClick(this, e);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Invoke the clear button click event
		/// </summary>
		/// <param name="e">Event args</param>
		protected virtual void OnClearBtnClick(EventArgs e) 
		{
			if (ClearBtnClick != null) 
			{
				ClearBtnClick(this, e);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Invoke the clear unit update event
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Event args</param>
		protected virtual void OnUpdateUnit(object sender, EventArgs e) 
		{
			if (UnitUpdate != null) 
			{
				UnitUpdate(sender, e);
			}
		}
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Invoke the search button click event
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Event args</param>
		protected virtual void OnSearchClick(object sender, System.EventArgs e)
		{
			//
		}
		
		#endregion

		#region Methods

		/// <summary>
		/// SetSelectButtonVisibility
		/// </summary>
		/// <param name="visible"></param>
		public void SetSelectButtonVisibility(bool visible)
		{
			this.btnSelectUnit.Visible = visible;
		}

		/// <summary>
		/// Sets the state of the Unit ID, Product ID, and Select unit button.
		/// </summary>
		/// <param name="enabled"></param>
		public void EnableTextBoxes(bool enabled)
		{
			this.txtUnitID.Enabled = enabled;
			this.txtProductID.Enabled = enabled;
			this.btnSelectUnit.Enabled = enabled;
		}


		/// <summary>
		/// Populates the control with blood unit and blood product data. 
		/// Performes the validation of the data.
		/// </summary>
		/// <param name="unitID"></param>
		/// <param name="fullProductCode"></param>
		public void PopulateSelector(string unitID, string fullProductCode)
		{
			txtProductID.Text = string.Empty;
			txtUnitID.Text = unitID;
			ValidateUnitID();

			if (fullProductCode.Trim().Length == 8)
			{
				txtProductID.Text = fullProductCode;
			}
			else
			{
				txtProductID.Text = fullProductCode.Substring(0,5);
			}
			ValidateProductCode();
		}

		/// <summary>
		/// Populates the control with blood unit and blood product data. 
		/// Performes the validation of the data.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void PopulateSelector(System.Guid bloodUnitGuid)
		{
			if ((this._currentBloodUnit == null) || (!System.Guid.Equals(this._currentBloodUnit.BloodUnitGuid,bloodUnitGuid)))
			{
				this._currentBloodUnit = new BOL.BloodUnit(bloodUnitGuid);
			}

			this.txtProductID.Text = this._currentBloodUnit.FullProductCode;
			this.txtUnitID.Text = this._currentBloodUnit.EyeReadableUnitId;

			this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
			this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***

			using (new WaitCursor())
			{
				UpdateGUIForBloodUnit(gov.va.med.vbecs.BOL.BloodUnit.GetBloodUnitDetails(this._currentBloodUnit.LabelUnitId, this._currentBloodUnit.BloodProductGuid));
			}

		}

		/// <summary>
		/// Allows the Data to be refreshed and repopulated (implemented for UC26)
		/// </summary>
		public void RefreshBloodUnitData()
		{
			if (this._currentBloodUnit != null)
			{
				System.Guid unitGuid = this._currentBloodUnit.BloodUnitGuid;

				this._currentBloodUnit = null;
				this.PopulateSelector(unitGuid);
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		/// <summary>
		/// Public method to validate product code entered, and update blood bag details 
		/// if appropriate.
		/// </summary>
        public void ValidateProductCode()
        {
			bool errOnUnit = true;
			System.Guid bloodUnitGuid = Guid.Empty;
			DataTable selectedBloodUnitsTable;

			try
			{
				if (txtUnitID.Text == String.Empty && txtProductID.Text == String.Empty) 
				{
					this.Clear(false);
					return;
				}

				if (this.txtProductID.Text.Length == 0) return;
				if (!this._codabarProductCode.IsMatch(this.txtProductID.Text) && !this._isbtProductCode.IsMatch(this.txtProductID.Text))
				{
					errOnUnit = false;
					throw new BOL.BusinessObjectException("Unrecognized Product Code format");
				}

				if (this.txtUnitID.Text.Length == 0) return;
				if (!this._codabarUnitID.IsMatch(this.txtUnitID.Text) && !this._isbtUnitID.IsMatch(this.txtUnitID.Text))
				{
					throw new BOL.BusinessObjectException("Unrecognized Unit Id format");
				}

				if(txtProductID.Text.Length >= 5 && this.txtUnitID.Text.Length > 0)
				{

					// Handle possibility of multiple units returned
					selectedBloodUnitsTable = BOL.BloodUnit.GetBloodUnitsByIdAndProduct(this.txtUnitID.Text, this.txtProductID.Text.Substring(0,5), Common.LogonUser.LogonUserDivisionCode, BOL.BloodUnit.GetUnitDivisionId(this.txtProductID.Text), BOL.BloodUnit.GetDonationTypeCode(this.txtProductID.Text), this.AdditionalSearchCriteria);
					//CR3294 - Exception no longer thrown in GetBloodUnitsByIdAndProduct
                    if (selectedBloodUnitsTable.Rows.Count == 0)
                    {
                        throw new BOL.BusinessObjectException("BloodUnit not found for UnitId/Product Code");
                    }
                    //
				    if(selectedBloodUnitsTable.Rows.Count > 1)
					{ 
						DlgSelectBloodUnits unitSelector = new DlgSelectBloodUnits(selectedBloodUnitsTable, this.EnteredEyeReadableUnitId, this.EnteredUnitProductCode);
						unitSelector.MultiSelect = false;
						unitSelector.TitleBarText = "VBECS - Multiple Units Per Product Code - Please Select Unit";
						unitSelector.LogoBarText = "Select Unit";
						unitSelector.ShowDialog(this.FindForm());
						//
						if (unitSelector.DialogResult == DialogResult.OK)
						{
							bloodUnitGuid = (Guid)unitSelector.SelectedUnits.Rows[0][Common.VbecsTables.BloodUnit.BloodUnitGuid];
							if(unitSelector.SelectedUnits.Rows[0][Common.VbecsTables.BloodUnit.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive).ToString()))
							{
								Common.Utility.MessageBeep();
							}
						}
						else if(unitSelector.DialogResult == DialogResult.Cancel)
						{
							// user has cancelled so call clear event and return
							this.OnClearBtnClick(System.EventArgs.Empty);
							return;
						}
					}
					else
					{
						bloodUnitGuid = (Guid)selectedBloodUnitsTable.Rows[0][Common.VbecsTables.BloodUnit.BloodUnitGuid];
						if(selectedBloodUnitsTable.Rows[0][Common.VbecsTables.BloodUnit.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive).ToString()))
						{
							Common.Utility.MessageBeep();
						}
					}
					//								}

					if ((this._currentBloodUnit == null) || (!System.Guid.Equals(this._currentBloodUnit.BloodUnitGuid,bloodUnitGuid)))
					{
						this._currentBloodUnit = new BOL.BloodUnit(bloodUnitGuid);
					}


					this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
					this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***

					using (new WaitCursor())
					{
						UpdateGUIForBloodUnit(gov.va.med.vbecs.BOL.BloodUnit.GetBloodUnitDetails(this._currentBloodUnit.LabelUnitId, this._currentBloodUnit.BloodProductGuid));
					}
									
				} 
				
			}
			catch(Exception exp)
			{
				if (errOnUnit)
					this.errorProvider1.SetVbecsError(this.txtUnitID, exp.Message); // **** 508 Compliance ***
				else
					this.errorProvider1.SetVbecsError(this.txtProductID, exp.Message); // **** 508 Compliance ***
                
				this.Clear(true);
			}			

        }

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		/// <summary>
		/// Public method to validate unit id entered, and update blood bag details 
		/// if appropriate.
		/// </summary>
		public void ValidateUnitID()
		{
			System.Guid bloodUnitGuid = Guid.Empty;
			DataTable selectedBloodUnitsTable;
			bool errOnUnit = true;

			try
			{
				if (txtUnitID.Text == String.Empty && txtProductID.Text == String.Empty) 
				{
					this.Clear(false);
					return;
				}

				if (this.txtUnitID.Text == string.Empty) return;
				if (!this._codabarUnitID.IsMatch(this.txtUnitID.Text) && !this._isbtUnitID.IsMatch(this.txtUnitID.Text))
				{
					throw new BOL.BusinessObjectException("Unrecognized Unit Id format");
				}

				if (this.txtProductID.Text.Length == 0) return;
				if (!this._codabarProductCode.IsMatch(this.txtProductID.Text) && !this._isbtProductCode.IsMatch(this.txtProductID.Text))
				{
					errOnUnit = false;
					throw new BOL.BusinessObjectException("Unrecognized Product Code format");
				}
			

				if(txtProductID.Text.Length >= 5 && this.txtUnitID.Text.Length > 0)
				{

					selectedBloodUnitsTable = BOL.BloodUnit.GetBloodUnitsByIdAndProduct(this.txtUnitID.Text.Trim(), this.txtProductID.Text.Trim(), Common.LogonUser.LogonUserDivisionCode, BOL.BloodUnit.GetUnitDivisionId(this.txtProductID.Text), BOL.BloodUnit.GetDonationTypeCode(this.txtProductID.Text), this.AdditionalSearchCriteria);
                    //CR3294 - GetBloodUnitsByIdAndProduct() no longer throwing exception
                    if (selectedBloodUnitsTable.Rows.Count == 0)
                    {
                        throw new BOL.BusinessObjectException("BloodUnit not found for UnitId/Product Code");
                    }

					if(selectedBloodUnitsTable.Rows.Count > 1)
					{ 
						DlgSelectBloodUnits unitSelector = new DlgSelectBloodUnits(selectedBloodUnitsTable, this.EnteredEyeReadableUnitId, this.EnteredUnitProductCode);
						unitSelector.MultiSelect = false;
						unitSelector.TitleBarText = "VBECS - Multiple Units Per Product Code - Please Select Unit";
						unitSelector.LogoBarText = "Select Unit";
						unitSelector.ShowDialog(this.FindForm());
						//
						if (unitSelector.DialogResult == DialogResult.OK)
						{
							bloodUnitGuid = (Guid)unitSelector.SelectedUnits.Rows[0][Common.VbecsTables.BloodUnit.BloodUnitGuid];
							if(unitSelector.SelectedUnits.Rows[0][Common.VbecsTables.BloodUnit.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive).ToString()))
							{
								Common.Utility.MessageBeep();
							}
						}
						else if(unitSelector.DialogResult == DialogResult.Cancel)
						{
							// user has cancelled so call clear event and return
							OnClearBtnClick(EventArgs.Empty);
							return;
						}
					}
					else
					{
						bloodUnitGuid = (Guid)selectedBloodUnitsTable.Rows[0][Common.VbecsTables.BloodUnit.BloodUnitGuid];
						if(selectedBloodUnitsTable.Rows[0][Common.VbecsTables.BloodUnit.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive).ToString()))
						{
							Common.Utility.MessageBeep();
						}
					}
					//								}

					if ((this._currentBloodUnit == null) || (!System.Guid.Equals(this._currentBloodUnit.BloodUnitGuid,bloodUnitGuid)))
					{
						this._currentBloodUnit = new BOL.BloodUnit(bloodUnitGuid);
					}
                                
					using (new WaitCursor())
					{
						UpdateGUIForBloodUnit(gov.va.med.vbecs.BOL.BloodUnit.GetBloodUnitDetails(this._currentBloodUnit.LabelUnitId, this._currentBloodUnit.BloodProductGuid));
					}

					this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
					this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***

				}
				
			}
			catch(Exception exp)
			{
				if (errOnUnit)
                    this.errorProvider1.SetVbecsError(this.txtUnitID, exp.Message); // **** 508 Compliance ***
				else
					this.errorProvider1.SetVbecsError(this.txtProductID, exp.Message); // **** 508 Compliance ***

				this.Clear(true);
			}	
        }
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		/// <summary>
		/// Public method to clear blood bag details and display. Sets the properites
		/// of the control to initial values.
		/// </summary>
		public void Clear(bool keepUnitFields)
		{
			if (!keepUnitFields)
			{
                txtProductID.Text = String.Empty;
				txtUnitID.Text = String.Empty;

				//Reset the flags JIC
				this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
				this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***
			}

			btnSelect.Enabled = false;
			this.bloodUnitDetailsTab1.BloodUnit = null;
			this.bloodUnitLabel1.BloodUnit = null;

			OnClearBtnClick(EventArgs.Empty);
		}

		private void UpdateGUIForBloodUnit(DataTable dtTable)
		{
			using (new WaitCursor())
			{
				if(dtTable.Rows.Count > 0)
				{
					this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
					this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***
					btnSelect.Enabled = true;
				}
				else
				{
					btnSelect.Enabled = false;
					GuiMessenger.ShowMessageBox(Common.StrRes.ValidMsg.UC027.BR13UnitNotFound());
					this.txtUnitID.Focus();
					return;
				}

				try
				{
					this.bloodUnitDetailsTab1.BloodUnit = this._currentBloodUnit;
					this.bloodUnitLabel1.BloodUnit = this._currentBloodUnit;
					//
					if (this.txtUnitID.Text != this._currentBloodUnit.EyeReadableUnitId)
					{
						this.txtUnitID.Text = this._currentBloodUnit.EyeReadableUnitId;
					}
					//
					if (this.txtProductID.Text != this._currentBloodUnit.FullProductCode)
					{
						this.txtProductID.Text = this._currentBloodUnit.FullProductCode;
					}
					//
					OnUpdateUnit(this._currentBloodUnit, new EventArgs());
				}
				catch(Exception err)
				{
					this.Clear(true);
					this.errorProvider1.SetVbecsError(this.txtProductID, err.Message); // **** 508 Compliance ***
				}
			}
		}


		/// <summary>
		/// Sets the label and DetailsTab with info
		/// </summary>
		private void UpdateGUIForBloodUnit()
		{
			this.bloodUnitLabel1.BloodUnit = this._currentBloodUnit;
			this.bloodUnitDetailsTab1.BloodUnit = this._currentBloodUnit;
			if(this.txtUnitID.Text.Length > 0 && this.txtProductID.Text.Length > 0)
			{
				btnSelect.Enabled = this._currentBloodUnit.IsValid;
			}
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// Get the error messages active for the control
		/// </summary>
		/// <returns></returns>
		public string GetErrorMessages()
		{
			string textToSpeak = string.Empty;
			textToSpeak = this.bloodUnitDetailsTab1.GetErrorMessages();
			return textToSpeak = string.Concat(textToSpeak, "\n", this.errorProvider1.GetErrorMessages());
		}

		#endregion

		#region Control Events
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Key press event handler. Handles '~' character
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Key press event args</param>
		private new void KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			if (e.KeyChar == '~') 
			{
				e.Handled = true;
			}
		}
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Handles select unit button click event, invokes the serach click event
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Event args</param>
		private void btnSelectUnit_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();
			if (onSearchClick!=null) 
			{
				onSearchClick(sender,e);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Select button click event handler. Raises select button click event
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Event args</param>
		private void btnSelect_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();
			OnSelectBtnClick(EventArgs.Empty);
			Clear(false);
			// Can't put this line in the Clear method cos its called within the product id field
			this.txtUnitID.Focus();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Clear button click event handler
		/// </summary>
		/// <param name="sender">Sender object</param>
		/// <param name="e">Event args</param>
		private void btnClear_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();
			this.errorProvider1.SetVbecsError(this.txtProductID, string.Empty); // **** 508 Compliance ***
			this.errorProvider1.SetVbecsError(this.txtUnitID, string.Empty); // **** 508 Compliance ***
			Clear(false);
			// Can't put this line in the Clear method cos its called within the product id field
			this.txtUnitID.Focus();
		}

		private void BloodUnitSelector_Load(object sender, System.EventArgs e)
		{
			this.errorProvider1.SetIconAlignment(this.txtUnitID, ErrorIconAlignment.MiddleLeft);
			this.errorProvider1.SetIconAlignment(this.txtProductID, ErrorIconAlignment.MiddleLeft);
			this.txtUnitID.Focus();
		}

		private void txtUnitID_Leave(object sender, System.EventArgs e)
		{
			if (!this.RecursionGuard)
			{
				this.RecursionGuard = true;

				try
				{

					this.OnUserActivity();

					if (!this.txtUnitID.ReadOnly)
					{
						ValidateUnitID();
					}
					//
					if (this.btnSelect.Enabled)
					{
						this.btnSelect.Focus();
					}
				}
				finally
				{
					this.RecursionGuard = false;
				}
			}		
		}

		private void txtProductID_Leave(object sender, System.EventArgs e)
		{
			if (!this.RecursionGuard)
			{
				this.RecursionGuard = true;

				try
				{
					this.OnUserActivity();

					if (!this.txtProductID.ReadOnly)
					{
						this.ValidateProductCode();
					}
					//
					if (this.btnSelect.Enabled)
					{
						this.btnSelect.Focus();
					}
				}
				finally
				{
					this.RecursionGuard = false;
				}
			}		

		}

		/// <summary>
		/// Get/Set RecursionGuard
		/// </summary>
		public virtual bool RecursionGuard
		{
			get
			{
				return _recursionGuard;
			}
			set
			{
				this._recursionGuard = value;
			}
		}

		/// <summary>
		/// Handles KeyDown events for txtUnitID and txtProductID
		/// </summary>
		private void SearchKeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			this.OnUserActivity();

			if (!this.txtUnitID.ReadOnly)
			{
				if (e.KeyCode == Keys.Enter)
				{
					//Pretend ENTER is a tab
					this.SelectNextControl((TextBox)sender,true,true,true,true);
				}
			}
		}

		#endregion

	}
}
